<?php

namespace App\Console\Commands;

use App\ScheduledBroadcast;
use Huludini\PerfectWorldAPI\API;
use Illuminate\Console\Command;
use Carbon\Carbon;

class ProcessScheduledBroadcasts extends Command
{
    /**
     * The name and signature of the console command.
     *
     * @var string
     */
    protected $signature = 'broadcasts:process';

    /**
     * The console command description.
     *
     * @var string
     */
    protected $description = 'Process scheduled broadcasts that are due to be sent';

    /**
     * Create a new command instance.
     *
     * @return void
     */
    public function __construct()
    {
        parent::__construct();
    }

    /**
     * Execute the console command.
     *
     * @return int
     */
    public function handle()
    {
        // Check if scheduled broadcasts are enabled
        if (!settings('scheduled_broadcasts_enabled', true)) {
            $this->info('Scheduled broadcasts are disabled in settings.');
            return 0;
        }

        // Check if server is online
        if (!$this->isServerOnline()) {
            $this->info('Server is offline. Skipping scheduled broadcasts.');
            return 0;
        }

        $this->info('Processing scheduled broadcasts...');

        // Get broadcasts that are due to be sent
        $dueBroadcasts = ScheduledBroadcast::where('scheduled_at', '<=', Carbon::now())
            ->get();

        if ($dueBroadcasts->isEmpty()) {
            $this->info('No broadcasts due to be sent.');
            return 0;
        }

        $api = new API();
        $processed = 0;

        foreach ($dueBroadcasts as $broadcast) {
            try {
                // Send the broadcast
                $api->WorldChat($broadcast->user, $broadcast->message, $broadcast->channel);

                $this->info("Sent broadcast: {$broadcast->message} to user {$broadcast->user}");

                if ($broadcast->is_recurring) {
                    // Calculate next scheduled time
                    $nextScheduledAt = $this->calculateNextScheduledTime($broadcast);
                    $broadcast->scheduled_at = $nextScheduledAt;
                    $broadcast->save();

                    $this->info("Rescheduled recurring broadcast for: {$nextScheduledAt}");
                } else {
                    // Delete one-time broadcast
                    $broadcast->delete();
                    $this->info('Deleted one-time broadcast');
                }

                $processed++;
            } catch (\Exception $e) {
                $this->error("Failed to send broadcast ID {$broadcast->id}: {$e->getMessage()}");
            }
        }

        $this->info("Processed {$processed} broadcasts successfully.");
        return 0;
    }

    /**
     * Check if the game server is online
     *
     * @return bool
     */
    private function isServerOnline()
    {
        $api = new API();
        return $api->online;
    }

    /**
     * Calculate the next scheduled time for recurring broadcasts
     *
     * @param ScheduledBroadcast $broadcast
     * @return Carbon
     */
    private function calculateNextScheduledTime(ScheduledBroadcast $broadcast)
    {
        $scheduledAt = Carbon::parse($broadcast->scheduled_at);

        switch ($broadcast->recurrence_unit) {
            case 'minutes':
                return $scheduledAt->addMinutes($broadcast->recurrence_interval);
            case 'hours':
                return $scheduledAt->addHours($broadcast->recurrence_interval);
            case 'days':
                return $scheduledAt->addDays($broadcast->recurrence_interval);
            case 'weeks':
                return $scheduledAt->addWeeks($broadcast->recurrence_interval);
            case 'months':
                return $scheduledAt->addMonths($broadcast->recurrence_interval);
            default:
                return $scheduledAt->addDays(1); // Default to daily
        }
    }
}