<?php

namespace App\Http\Controllers\Admin;

use App\User;
use Illuminate\Http\Request;

use App\Http\Requests;
use App\Http\Controllers\Controller;

class MembersController extends Controller
{
    public function getManage()
    {
        pagetitle( [ trans( 'members.manage' ), trans( 'main.apps.members' ), settings( 'server_name' ) ] );
        $users = User::paginate();
        return view( 'admin.members.manage', compact( 'users' ) );
    }

    public function postBalance( Request $request, User $user )
    {
        $this->validate($request, [
            'amount' => 'required|numeric'
        ]);

        $user->money = $user->money + $request->amount;
        $user->save();

        flash()->success( trans( 'members.success', ['user' => $user->name, 'count' => $request->amount, 'currency' => strtolower( settings( 'currency_name' ) )] ) );
        return redirect()->back();
    }

    public function postSearch( Request $request )
    {
        return User::where( 'name', 'LIKE', '%' . $request->search_query . '%' )->get();
    }

    public function postRole(Request $request, User $user)
    {
        $this->validate($request, [
            'role' => 'required|in:member,game master,administrator'
        ]);

        $user->role = $request->role;
        $user->save();

        if ($request->ajax()) {
            return response()->json(['status' => 'ok', 'role' => $user->role]);
        }

        flash()->success(trans('members.role_updated', ['user' => $user->name, 'role' => $user->role]));
        return redirect()->back();
    }

    /**
     * Update user profile fields from the manage UI.
     * Accepts: name, email, role, balance (optional: sets absolute balance if provided)
     */
    public function postUpdate(Request $request, User $user)
    {
        $this->validate($request, [
            'name' => 'required|string|max:255',
            'email' => 'required|email|max:255',
            'role' => 'required|in:member,game master,administrator',
            'balance' => 'nullable|numeric'
        ]);

        $user->name = $request->name;
        $user->email = $request->email;
        $user->role = $request->role;

        if ($request->has('balance') && strlen(trim((string)$request->balance)) > 0) {
            // Set absolute balance
            $user->money = $request->balance;
        }

        $user->save();

        if ($request->ajax()) {
            return response()->json(['status' => 'ok', 'user' => $user]);
        }

        flash()->success(trans('members.updated', ['user' => $user->name]));
        return redirect()->back();
    }

    /**
     * Update user's email via AJAX or form post
     */
    public function postEmail(Request $request, User $user)
    {
        $this->validate($request, [
            'email' => 'required|email|max:255'
        ]);

        $user->email = $request->email;
        $user->save();

        if ($request->ajax()) {
            return response()->json(['status' => 'ok', 'user' => $user]);
        }

        flash()->success(trans('members.email_updated', ['user' => $user->name]));
        return redirect()->back();
    }

    /**
     * Update user's password via AJAX or form post
     */
    public function postPassword(Request $request, User $user)
    {
        $this->validate($request, [
            'password' => 'required|min:6|confirmed'
        ]);

        $user->password = bcrypt($request->password);
        $user->save();

        if ($request->ajax()) {
            return response()->json(['status' => 'ok']);
        }

        flash()->success(trans('members.password_updated', ['user' => $user->name]));
        return redirect()->back();
    }
}
