<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\WikiArticle;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Str;

class WikiController extends Controller
{
    /**
     * Display wiki articles list
     */
    public function getIndex()
    {
        $articles = WikiArticle::orderBy('created_at', 'desc')->paginate(15);
        return view('admin.wiki.index', compact('articles'));
    }

    /**
     * Show create article form
     */
    public function getCreate()
    {
        $categories = WikiArticle::getCategories();
        return view('admin.wiki.create', compact('categories'));
    }

    /**
     * Store new article
     */
    public function postCreate(Request $request)
    {
        $this->validate($request, [
            'title' => 'required|max:255',
            'content' => 'required',
            'category' => 'required|in:' . implode(',', array_keys(WikiArticle::getCategories())),
            'status' => 'required|in:published,draft'
        ]);

        $article = new WikiArticle();
        $article->title = $request->title;
        $article->slug = Str::slug($request->title);
        $article->content = $request->content;
        $article->excerpt = $request->excerpt;
        $article->category = $request->category;
        $article->status = $request->status;
        $article->author_id = Auth::id();
        
        // Ensure unique slug
        $originalSlug = $article->slug;
        $counter = 1;
        while (WikiArticle::where('slug', $article->slug)->exists()) {
            $article->slug = $originalSlug . '-' . $counter;
            $counter++;
        }
        
        $article->save();

        return redirect('admin/wiki')->with('success', trans('wiki.create_success'));
    }

    /**
     * Show edit article form
     */
    public function getEdit($id)
    {
        $article = WikiArticle::findOrFail($id);
        $categories = WikiArticle::getCategories();
        return view('admin.wiki.edit', compact('article', 'categories'));
    }

    /**
     * Update article
     */
    public function postEdit(Request $request, $id)
    {
        $this->validate($request, [
            'title' => 'required|max:255',
            'content' => 'required',
            'category' => 'required|in:' . implode(',', array_keys(WikiArticle::getCategories())),
            'status' => 'required|in:published,draft'
        ]);

        $article = WikiArticle::findOrFail($id);
        $article->title = $request->title;
        $article->content = $request->content;
        $article->excerpt = $request->excerpt;
        $article->category = $request->category;
        $article->status = $request->status;
        
        // Update slug if title changed
        $newSlug = Str::slug($request->title);
        if ($article->slug !== $newSlug) {
            $originalSlug = $newSlug;
            $counter = 1;
            while (WikiArticle::where('slug', $newSlug)->where('id', '!=', $id)->exists()) {
                $newSlug = $originalSlug . '-' . $counter;
                $counter++;
            }
            $article->slug = $newSlug;
        }
        
        $article->save();

        return redirect('admin/wiki')->with('success', trans('wiki.edit_success'));
    }

    /**
     * Delete article
     */
    public function getDelete($id)
    {
        $article = WikiArticle::findOrFail($id);
        $article->delete();

        return redirect('admin/wiki')->with('success', trans('wiki.delete_success'));
    }
}