<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use Illuminate\Http\Response;
use Illuminate\Support\Facades\Session;

class CaptchaController extends Controller
{
    public function generate()
    {
        // Check if GD extension is available
        if (!extension_loaded('gd')) {
            return $this->generateSvgCaptcha();
        }
        
        try {
            // Generate random captcha text
            $characters = '23456789ABCDEFGHJKLMNPQRSTUVWXYZ';
            $captchaText = '';
            for ($i = 0; $i < 5; $i++) {
                $captchaText .= $characters[rand(0, strlen($characters) - 1)];
            }
            
            // Store in session
            Session::put('captcha', strtolower($captchaText));
            
            // Create image
            $width = 150;
            $height = 50;
            $image = imagecreate($width, $height);
            
            // Colors
            $bgColor = imagecolorallocate($image, 45, 55, 72); // Dark background
            $textColor = imagecolorallocate($image, 255, 255, 255); // White text
            $lineColor = imagecolorallocate($image, 100, 116, 139); // Gray lines
            
            // Fill background
            imagefill($image, 0, 0, $bgColor);
            
            // Add noise lines
            for ($i = 0; $i < 8; $i++) {
                imageline($image, rand(0, $width), rand(0, $height), rand(0, $width), rand(0, $height), $lineColor);
            }
            
            // Add text with better positioning
            $fontSize = 5;
            $charWidth = imagefontwidth($fontSize);
            $charHeight = imagefontheight($fontSize);
            
            for ($i = 0; $i < strlen($captchaText); $i++) {
                $charX = 15 + ($i * ($charWidth + 8)) + rand(-3, 3);
                $charY = ($height - $charHeight) / 2 + rand(-5, 5);
                imagestring($image, $fontSize, $charX, $charY, $captchaText[$i], $textColor);
            }
            
            // Output image
            ob_start();
            imagepng($image);
            $imageData = ob_get_contents();
            ob_end_clean();
            imagedestroy($image);
            
            return response($imageData, 200, [
                'Content-Type' => 'image/png',
                'Cache-Control' => 'no-cache, no-store, must-revalidate',
                'Pragma' => 'no-cache',
                'Expires' => '0'
            ]);
            
        } catch (Exception $e) {
            // Fallback to SVG if GD fails
            return $this->generateSvgCaptcha();
        }
    }
    
    private function generateSvgCaptcha()
    {
        // Generate random captcha text
        $characters = '23456789ABCDEFGHJKLMNPQRSTUVWXYZ';
        $captchaText = '';
        for ($i = 0; $i < 5; $i++) {
            $captchaText .= $characters[rand(0, strlen($characters) - 1)];
        }
        
        // Store in session
        Session::put('captcha', strtolower($captchaText));
        
        // Create SVG
        $svg = '<svg width="150" height="50" xmlns="http://www.w3.org/2000/svg">';
        $svg .= '<rect width="150" height="50" fill="#2d3748"/>';
        
        // Add noise lines
        for ($i = 0; $i < 8; $i++) {
            $x1 = rand(0, 150);
            $y1 = rand(0, 50);
            $x2 = rand(0, 150);
            $y2 = rand(0, 50);
            $svg .= '<line x1="' . $x1 . '" y1="' . $y1 . '" x2="' . $x2 . '" y2="' . $y2 . '" stroke="#64748b" stroke-width="1"/>';
        }
        
        // Add text
        for ($i = 0; $i < strlen($captchaText); $i++) {
            $x = 15 + ($i * 25) + rand(-3, 3);
            $y = 30 + rand(-5, 5);
            $svg .= '<text x="' . $x . '" y="' . $y . '" font-family="monospace" font-size="20" font-weight="bold" fill="#ffffff">' . $captchaText[$i] . '</text>';
        }
        
        $svg .= '</svg>';
        
        return response($svg, 200, [
            'Content-Type' => 'image/svg+xml',
            'Cache-Control' => 'no-cache, no-store, must-revalidate',
            'Pragma' => 'no-cache',
            'Expires' => '0'
        ]);
    }
    
    public function refresh(Request $request)
    {
        $answer = $request->input('answer');
        Session::put('captcha', $answer);
        return response()->json(['success' => true]);
    }
    
    public function refreshReg(Request $request)
    {
        $answer = $request->input('answer');
        Session::put('reg_captcha', $answer);
        return response()->json(['success' => true]);
    }
    
    public static function validateCaptcha($input)
    {
        $sessionCaptcha = Session::get('captcha');
        Session::forget('captcha');
        
        return $sessionCaptcha && (int)$input === (int)$sessionCaptcha;
    }
    
    public static function validateRegCaptcha($input)
    {
        $sessionCaptcha = Session::get('reg_captcha');
        Session::forget('reg_captcha');
        
        return $sessionCaptcha && (int)$input === (int)$sessionCaptcha;
    }
}