<?php

namespace App\Http\Controllers\Front;

use App\Http\Controllers\Controller;
use App\WikiArticle;
use Illuminate\Http\Request;

class WikiController extends Controller
{
    /**
     * Display wiki home page with categories
     */
    public function getIndex(Request $request)
    {
        $category = $request->get('category');
        $search = $request->get('search');
        
        $query = WikiArticle::published()->orderBy('created_at', 'desc');
        
        if ($category) {
            $query->category($category);
        }
        
        if ($search) {
            $query->where(function($q) use ($search) {
                $q->where('title', 'like', '%' . $search . '%')
                  ->orWhere('content', 'like', '%' . $search . '%');
            });
        }
        
        $articles = $query->paginate(12);
        $categories = WikiArticle::getCategories();
        
        // Get article counts per category
        $categoryCounts = [];
        foreach ($categories as $key => $name) {
            $categoryCounts[$key] = WikiArticle::published()->category($key)->count();
        }
        
        return view('front.wiki.index', compact('articles', 'categories', 'categoryCounts', 'category', 'search'));
    }

    /**
     * Display specific wiki article
     */
    public function getArticle($slug)
    {
        $article = WikiArticle::published()->where('slug', $slug)->firstOrFail();
        
        // Increment view count
        $article->incrementViews();
        
        // Get related articles from same category
        $relatedArticles = WikiArticle::published()
            ->category($article->category)
            ->where('id', '!=', $article->id)
            ->orderBy('created_at', 'desc')
            ->limit(5)
            ->get();
        
        return view('front.wiki.article', compact('article', 'relatedArticles'));
    }

    /**
     * Display articles by category
     */
    public function getCategory($category)
    {
        $categories = WikiArticle::getCategories();
        
        if (!array_key_exists($category, $categories)) {
            abort(404);
        }
        
        $articles = WikiArticle::published()
            ->category($category)
            ->orderBy('created_at', 'desc')
            ->paginate(12);
        
        $categoryName = $categories[$category];
        
        return view('front.wiki.category', compact('articles', 'category', 'categoryName', 'categories'));
    }
}