<?php

namespace App;

use Illuminate\Database\Eloquent\Model;
use Illuminate\Support\Str;

class WikiArticle extends Model
{
    protected $fillable = [
        'title', 'slug', 'content', 'excerpt', 'category', 'status', 'author_id'
    ];

    protected $dates = ['created_at', 'updated_at'];

    /**
     * Get the author of the article
     */
    public function author()
    {
        return $this->belongsTo(User::class, 'author_id');
    }

    /**
     * Generate slug from title
     */
    public function setTitleAttribute($value)
    {
        $this->attributes['title'] = $value;
        $this->attributes['slug'] = Str::slug($value);
    }

    /**
     * Scope for published articles
     */
    public function scopePublished($query)
    {
        return $query->where('status', 'published');
    }

    /**
     * Scope for specific category
     */
    public function scopeCategory($query, $category)
    {
        return $query->where('category', $category);
    }

    /**
     * Get available categories
     */
    public static function getCategories()
    {
        return [
            'general' => 'General',
            'gameplay' => 'Gameplay',
            'classes' => 'Classes',
            'items' => 'Items & Equipment',
            'quests' => 'Quests & Missions',
            'pvp' => 'PvP & Territory Wars',
            'guilds' => 'Guilds & Factions',
            'events' => 'Events',
            'guides' => 'Guides & Tutorials',
            'faq' => 'FAQ'
        ];
    }

    /**
     * Get category display name
     */
    public function getCategoryNameAttribute()
    {
        $categories = self::getCategories();
        return $categories[$this->category] ?? ucfirst($this->category);
    }

    /**
     * Increment view count
     */
    public function incrementViews()
    {
        $this->increment('views');
    }

    /**
     * Get excerpt or generate from content
     */
    public function getExcerptAttribute($value)
    {
        if ($value) {
            return $value;
        }
        
        return Str::limit(strip_tags($this->content), 200);
    }
}