<?php
/**
 * Utility script to help update admin pages from old Bootstrap/Metronic styling to modern dark theme
 * This script provides templates and patterns for updating admin pages
 */

// Common patterns to replace in admin pages:

// 1. Basic page wrapper
$oldWrapper = '@extends(\'admin.layouts.app\')

@section( \'content\' )
    <div class="portlet light bordered">
        <div class="portlet-body form">';

$newWrapper = '@extends(\'admin.layouts.app\')

@section(\'content\')
<div class="max-w-4xl mx-auto">
    <div class="dark-card">
        <div class="px-6 py-4" style="border-bottom: 1px solid #374151;">
            <div class="flex items-center justify-between">
                <div>
                    <h1 class="text-2xl font-bold text-white">PAGE_TITLE</h1>
                    <p class="text-gray-400 mt-1">PAGE_DESCRIPTION</p>
                </div>
                <div class="w-12 h-12 bg-gradient-to-br from-ICON_COLOR-600 to-ICON_COLOR-600 rounded-lg flex items-center justify-center">
                    <i class="fas fa-ICON_NAME text-white"></i>
                </div>
            </div>
        </div>
        <div class="p-6">';

// 2. Form wrapper
$oldForm = '<form action="URL" method="post" class="form-horizontal">
                {!! csrf_field() !!}
                <div class="form-body">';

$newForm = '<form action="URL" method="post" class="space-y-6">
                {!! csrf_field() !!}';

// 3. Form group
$oldFormGroup = '<div class="form-group form-md-line-input">
                        <label class="col-md-2 control-label" for="FIELD_ID">LABEL</label>
                        <div class="col-md-9">
                            <input name="FIELD_NAME" type="FIELD_TYPE" class="form-control" id="FIELD_ID" value="FIELD_VALUE">
                            <div class="form-control-focus"> </div>
                            <span class="help-block">HELP_TEXT</span>
                        </div>
                    </div>';

$newFormGroup = '<div>
                    <label for="FIELD_ID" class="block text-sm font-medium text-gray-300 mb-2">
                        LABEL <span class="text-red-400">*</span>
                    </label>
                    <input name="FIELD_NAME" 
                           type="FIELD_TYPE" 
                           class="w-full" 
                           id="FIELD_ID" 
                           value="FIELD_VALUE"
                           required>
                    <p class="mt-1 text-sm text-gray-400">HELP_TEXT</p>
                </div>';

// 4. Form actions
$oldActions = '<div class="form-actions">
                    <div class="row">
                        <div class="col-md-offset-2 col-md-9">
                            <button type="submit" class="btn green">BUTTON_TEXT</button>
                        </div>
                    </div>
                </div>';

$newActions = '<div class="flex justify-end pt-6" style="border-top: 1px solid #374151;">
                    <button type="submit" class="btn-primary inline-flex items-center">
                        <i class="fas fa-save mr-2"></i>BUTTON_TEXT
                    </button>
                </div>';

// 5. Page closing
$oldClosing = '</div>
    </div>
@endsection';

$newClosing = '</div>
    </div>
</div>
@endsection';

// This script performs safe, reversible conversions of old Bootstrap/Metronic admin
// Blade views into Tailwind-friendly markup. It backs up each modified file with
// a "-bootstrap" suffix so changes can be reviewed and reverted.

// Usage: php update_admin_pages.php

$base = __DIR__ . '/resources/views/admin';
if (!is_dir($base)) {
    echo "Admin views directory not found: $base\n";
    exit(1);
}

// Replacement rules: simple, conservative patterns to avoid breaking Blade logic.
$replacements = [
    // portlet wrappers -> dark-card container
    '/<div class="col-md-\d+">\s*<div class="portlet[^"]*">/i' => '<div class="dark-card">',
    '/<div class="portlet\b[^>]*>\s*<div class="portlet-title">[\s\S]*?<\/div>\s*<div class="portlet-body">/i' => '<div class="dark-card">',
    '/<div class="portlet\b[^>]*>\s*<div class="portlet-body">/i' => '<div class="dark-card">',

    // close portlet blocks
    '/<\/div>\s*<\/div>\s*<\/div>/' => '</div>',

    // form-horizontal -> vertical spacing
    '/class="form-horizontal"/i' => 'class="space-y-6"',

    // Bootstrap form groups -> simpler block with label/input classes (conservative)
    '/<div class="form-group[^"]*">([\s\S]*?)<\/div>\s*<\/div>/i' => '$1</div>',

    // form-control -> keep element but remove class references to Bootstrap
    '/class="([^"]*)form-control([^"]*)"/i' => 'class="$1w-full py-2 px-3 rounded-md bg-gray-800 border border-gray-700 text-gray-200$2"',

    // form-control-focus and help-block -> convert to helper paragraph
    '/<div class="form-control-focus">\s*<\/div>/i' => '',
    '/<span class="help-block">([\s\S]*?)<\/span>/i' => '<p class="mt-1 text-sm text-gray-400">$1</p>',

    // btn green -> Tailwind primary button class
    '/class="([^"]*)btn\s+green([^"]*)"/i' => 'class="$1inline-flex items-center px-4 py-2 rounded-md bg-indigo-600 hover:bg-indigo-500 text-white font-semibold$2"',
    '/class="([^"]*)btn\s+green-haze([^"]*)"/i' => 'class="$1inline-flex items-center px-3 py-2 rounded-md bg-green-600 hover:bg-green-500 text-white font-semibold$2"',

    // col-md-* wrappers: remove and let Tailwind layout handle spacing
    '/<div class="col-md-\d+">/i' => '<div>',
    '/<div class="col-md-offset-\d+\s+col-md-\d+">/i' => '<div class="flex justify-end">',
];

// Find all blade files under admin
$rii = new RecursiveIteratorIterator(new RecursiveDirectoryIterator($base));
$files = [];
foreach ($rii as $file) {
    if ($file->isDir()) continue;
    if (preg_match('/\\.blade\\.php$/', $file->getFilename())) {
        $files[] = $file->getPathname();
    }
}

if (empty($files)) {
    echo "No admin Blade files found to update.\n";
    exit(0);
}

$report = [];
foreach ($files as $file) {
    $orig = file_get_contents($file);
    $modified = $orig;

    foreach ($replacements as $pattern => $replacement) {
        $new = preg_replace($pattern, $replacement, $modified);
        if ($new !== null) {
            $modified = $new;
        }
    }

    if ($modified !== $orig) {
        // backup original
        $backup = $file . '-bootstrap';
        if (!file_exists($backup)) {
            copy($file, $backup);
        }
        file_put_contents($file, $modified);
        $report[] = $file;
        echo "Updated: $file\n";
    }
}

echo "\nConversion complete. Files updated: " . count($report) . "\n";
if (!empty($report)) {
    echo "Backups saved alongside original files with '-bootstrap' suffix.\n";
}

echo "\nNote: This script makes conservative regex replacements. Review each changed file and run tests.\n";
?>